implementation module layout


//	Clean Object I/O library, version 1.1


import	StdInt, StdBool, StdFunc, StdList, StdTuple
import	commondef, id


layoutError :: String String -> .x
layoutError rule error
	=	Error rule "layout" error

layoutFatalError :: String String -> .x
layoutFatalError rule error
	=	FatalError rule "layout" error


/*	Calculate the precise position (in pixels) of each ItPos element.
	The position is calculated from a zero origin.
	Assumptions:
	-	All ItPos elements have a layout element ItemPos.
	-	All relative references to previous elements have been identified (so LeftOfPrev --> LeftOf id and so on).
*/
calcItemPositions :: !(!Int,!Int) !(!Int,!Int) !(!Int,!Int) !Size !Size !Point ![ItPos] -> (!Size,![ItPos])
calcItemPositions hMargins=:(lMargin,rMargin) vMargins=:(tMargin,bMargin) itemSpaces reqSize minSize origin itPoss
	# itPoss	= sortItemPositions itPoss
	# (_,itPoss)= StateMap2 (calcItemPosition itemSpaces origin) itPoss (0,[])
	  size		= calcAreaSize itPoss reqSize minSize
	# itPoss	= StateMap2 (renterCornerItem hMargins vMargins size) itPoss []
	= ({w=lMargin+size.w+rMargin,h=tMargin+size.h+bMargin},itPoss)


/*	sortItemPositions sorts the list of item positions such that relatively laynout items 
	are placed immediately(!!) behind their target items.
	sortItemPositions failures:
	-	a cyclic dependency has been located: the Ids are printed and computation stops
	-	unknown references have been located: the Ids are printed and computation stops
*/
sortItemPositions :: ![ItPos] -> [ItPos]
sortItemPositions itPoss
	# (lineItems,relItems)	= divide (\{ipPos}->IsLine ipPos) itPoss
	# itPoss				= lineItems++relItems
	# itPoss				= sortItemPositions` [] itPoss
	= itPoss
where
	sortItemPositions` :: ![ItPos] ![ItPos] -> [ItPos]
	sortItemPositions` done todo
		| isEmpty todo
		= reverse done
		# (item1,todo)		= HdTl todo
		  pos1				= item1.ipPos
		  (isRelative,id2)	= IsRelative pos1
		| not isRelative
		= sortItemPositions` [item1:done] todo
		# (done,chain,todo)	= getItemPosChain id2 done [item1] todo
		= sortItemPositions` (insertItemPosChain chain done) todo
	where
		getItemPosChain :: !Id ![ItPos] ![ItPos] ![ItPos] -> (![ItPos],![ItPos],![ItPos])
		getItemPosChain nextId done chain todo
			# in_chain				= ipContainsId nextId chain
			| in_chain
			= layoutError "calculating layout" ("cyclic dependency between Ids: "+++listToString (map (\{ipId}->ipId) (reverse chain)) nextId)
			# in_done				= ipContainsId nextId done
			| in_done
			= (done,chain,todo)
			# (in_todo,next,todo)	= ipRemoveItPos nextId todo
			| not in_todo
			= layoutError "calculating layout" ("reference to unknown Id: "+++toString nextId)
			# nextPos				= next.ipPos
			  (isRelative,id2)		= IsRelative nextPos
			| not isRelative
			= (done,[next:chain],todo)
			= getItemPosChain id2 done [next:chain] todo
		where
			listToString :: ![x] !x -> {#Char} | toString x
			listToString xs x
				= "["+++foldr (\a b->toString a+++","+++toString b) "" xs+++toString x+++"]"
		
		insertItemPosChain :: ![ItPos] ![ItPos] -> [ItPos]
		insertItemPosChain chain=:[finalItPos:_] done
			| not isRelative
			= chain`++done
			= insertchain id chain` done
		where
			(isRelative,id)	= IsRelative finalItPos.ipPos
			chain`			= reverse chain
			
			insertchain :: !Id [ItPos] ![ItPos] -> [ItPos]
			insertchain id chain [ip:ips]
				| id==ip.ipId
				= chain++[ip:ips]
				= [ip:insertchain id chain ips]
			insertchain _ chain _
				= chain
		insertItemPosChain _ done				// this alternative will actually never be reached
			= done
	
	divide :: !(Cond x) ![x] -> (![x],![x])		// divide cond xs = (filter cond xs,filter (not o cond) xs)
	divide f [x:xs]
		| f x
		= ([x:yes],no)
		= (yes,[x:no])
	where
		(yes,no)= divide f xs
	divide _ _
		= ([],[])


/*	Calculate the positions of line oriented items and the space they occupy. 
	Place relatively placed items in the dependency list of the item referred to.
	Relatively placed items with unknown reference are transformed to (Left,zero) items.
	Items that are positioned at a fixed spot (Fix pos) are layn out relative to the given origin.
	Assumptions:
	-	All relative layout positions refer to existing elements which must occur in the done list.
	
	Note:	Renter = Right or Center,
			Corner = LeftTop, RightTop, LeftBottom or RightBottom
*/
calcItemPosition :: !(!Int,!Int) !Point !ItPos !(!Int,![ItPos]) -> (!Int,![ItPos])
calcItemPosition itemSpaces origin item1 sDone=:(sizeY,done)
	| isFix
	= (sizeY, [{item1 & ipCorner=fixpos-origin}:done])
	| isRelative && exists
	= (sizeY`,[item2`:done1])
	with
		(sizeY`,item2`)	= if (IsRelativeX pos1)
							 (calcXPosition itemSpaces id2 item1 sizeY item2)
							 (calcYPosition itemSpaces id2 item1 sizeY item2)
	| isRelative
	= layoutFatalError "calculating layout" "reference to unknown Id (not caught by sortItemPositions)"
	| IsCorner pos1
	= (sizeY, [item1:done])
	= (max sizeY (sizeY+yOffset1+h), [ipShift {vx=0,vy=sizeY+yOffset1} item1:done])	// PA: first tuple result was sizeY+yOffset1+h, now max is taken with previous sizeY value.
	with
		h				= item1.ipSize.h
		(_,offset)		= pos1
		yOffset			= offset.vy
		yOffset1		= if (sizeY==0) yOffset (snd itemSpaces+yOffset)
where
	pos1				= item1.ipPos
	(isFix,fixpos)		= IsFix pos1
	(isRelative,id2)	= IsRelative pos1
	(exists,item2,done1)= ipRemoveItPos id2 done
	
	//	calcXPosition calculates the position of item1 which is horizontally relative to the item identified by id2.
	//	This item is either item2 or occurs in the dependency list of item2.
	//	item1 is placed in the dependency list of item2.
	calcXPosition :: !(!Int,!Int) !Id !ItPos !Int !ItPos -> (!Int,!ItPos)
	calcXPosition itemSpaces id2 item1=:{ipId=id1,ipOffset=ipoffset1,ipPos=pos1,ipSize=size1,ipCorner=corner1} sizeY
								 item2=:{ipPos=pos2,ipDepends=depends2}
		= (	if (IsCorner pos2) sizeY (max (t+tShift+size1.h) sizeY)
		  ,	shift {item2 & ipDepends = [depend:depends2]}
		  )
	where
		xOffset1		= max 0 (fst itemSpaces+offset.vx)
		l				= if (IsLeftOf pos1) (corner2.x-size1.w-xOffset1) (corner2.x+size2.w+xOffset1)
		t				= corner2.y+offset.vy
		lShift			= shiftToZero l
		tShift			= shiftToZero t
		shiftVector		= {vx=lShift,vy=tShift}
		shift			= if (lShift>0 || tShift>0) (ipShift shiftVector) id
		(_,offset)		= pos1
		(corner2,size2)	= ipGetItemRect id2 item2
		depend			= cpShift {vx=l-corner1.x,vy=t-corner1.y} {cpId=id1,cpCorner=corner1,cpSize=size1,cpOffset=ipoffset1}
//		depend			= {cpId=id1,cpCorner={x=l,y=t},cpSize=size1,cpOffset=ipoffset1+{vx=l,vy=t}}
	
	//	calcYPosition calculates the position of item1 which is vertically relative to the item identified by id2.
	//	This item is either item2 or occurs in the dependency list of item2.
	//	item1 is placed in the dependency list of item2.
	calcYPosition :: !(!Int,!Int) !Id !ItPos !Int !ItPos -> (!Int,!ItPos)
	calcYPosition itemSpaces id2 item1=:{ipId=id1,ipOffset=ipoffset1,ipPos=pos1,ipSize=size1,ipCorner=corner1} sizeY
								 item2=:{ipPos=pos2,ipDepends=depends2}
		= (	if (IsCorner pos2) sizeY (max (t+tShift+size1.h) sizeY)
		  ,	shift {item2 & ipDepends = [depend:depends2]}
		  )
	where
		yOffset1		= max 0 (snd itemSpaces+offset.vy)
		l				= corner2.x+offset.vx
		t				= if (IsBelow pos1) (corner2.y+size2.h+yOffset1) (corner2.y-size1.h-yOffset1)
		lShift			= shiftToZero l
		tShift			= shiftToZero t
		shiftVector		= {vx=lShift,vy=tShift}
		shift			= if (lShift>0 || tShift>0) (ipShift shiftVector) id
		(_,offset)		= pos1
		(corner2,size2)	= ipGetItemRect id2 item2
		depend			= cpShift {vx=l-corner1.x,vy=t-corner1.y} {cpId=id1,cpCorner=corner1,cpSize=size1,cpOffset=ipoffset1}
//		depend			= {cpId=id1,cpCorner={x=l,y=t},cpSize=size1,cpOffset=ipoffset1+{vx=l,vy=t}}
	
	shiftToZero :: !Int -> Int
	shiftToZero x
		| x<0
		= 0-x
		= 0


/*	In case no requested size is given (requested size==zero), calculate the actual 
	width and height of the overall area. The overall area is the smallest enclosing 
	rectangle of the line oriented items, provided it fits the corner oriented items.
	In case of a requested size, yield this size.
*/
calcAreaSize :: ![ItPos] !Size !Size -> Size
calcAreaSize ips {w=0,h=0} {w=minW,h=minH}
	= {w=max areasize.w minW,h=max areasize.h minH}
where
	areasize		= StateMap2 calcItemAreaSize ips zero
	
	calcItemAreaSize :: !ItPos !Size -> Size
	calcItemAreaSize item size
		= extendAreaSize pos sizeItem size
	where
		pos			= item.ipPos
		depends		= item.ipDepends
		ipCorner	= item.ipCorner
		ipSize		= item.ipSize
		sizeItem	= cpMaxXY {w=ipCorner.x+ipSize.w,h=ipCorner.y+ipSize.h} depends
		
		extendAreaSize :: !ItemPos !Size !Size -> Size
		extendAreaSize (LeftTop,	{vx,vy}) {w=itemR,h=itemB} {w=sizeX,h=sizeY} = {w=max sizeX (itemR+vx),h=max sizeY (itemB+vy)}
		extendAreaSize (RightTop,	{vx,vy}) {w=itemR,h=itemB} {w=sizeX,h=sizeY} = {w=max sizeX (itemR-vx),h=max sizeY (itemB+vy)}
		extendAreaSize (LeftBottom,	{vx,vy}) {w=itemR,h=itemB} {w=sizeX,h=sizeY} = {w=max sizeX (itemR+vx),h=max sizeY (itemB-vy)}
		extendAreaSize (RightBottom,{vx,vy}) {w=itemR,h=itemB} {w=sizeX,h=sizeY} = {w=max sizeX (itemR-vx),h=max sizeY (itemB-vy)}
		extendAreaSize (_,			{vx})	 {w=itemR,h=itemB} {w=sizeX,h=sizeY} = {w=max sizeX (itemR+vx),h=max sizeY itemB}
calcAreaSize _ {w=reqW,h=reqH} {w=minW,h=minH}
	= {w=max minW reqW,h=max minH reqH}


/*	Position Renter/Corner items and adjust all elements to margins. */

renterCornerItem :: !(!Int,Int) !(!Int,Int) !Size !ItPos ![ItPos] -> [ItPos]
renterCornerItem hMargins=:(lMargin,_) vMargins=:(tMargin,_) sizeArea=:{w=width,h=height} item done
	| IsRenter pos || IsCorner pos
	= [ipShift shift item:done]
	with
		depends		= item.ipDepends
		ipCorner	= item.ipCorner
		ipSize		= item.ipSize
		sizeItem	= cpMaxXY {w=ipCorner.x+ipSize.w,h=ipCorner.y+ipSize.h} depends
		widthLeft	= width-sizeItem.w
		v			= if (IsCorner pos)
						 (cornerShift  pos sizeItem sizeArea)
						 {vx=lineShift pos widthLeft,vy=0}
		shift		= {vx=v.vx+lMargin,vy=v.vy+tMargin}
		
		lineShift :: !ItemPos !Int -> Int
		lineShift (Center,{vx}) space = space/2+vx
		lineShift (_,	  {vx}) space = space  +vx
		
		cornerShift :: !ItemPos !Size !Size -> ItemOffset
		cornerShift (LeftTop,	 v) _ _ = v
		cornerShift (RightTop,	 v=:{vx,vy}) {w=wItem}         {w}   = {v & vx=w-wItem+vx}
		cornerShift (LeftBottom, v=:{vx,vy}) {h=hItem}         {h}   = {v & vy=h-hItem+vy}
		cornerShift (RightBottom,   {vx,vy}) {w=wItem,h=hItem} {w,h} = {vx=w-wItem+vx,vy=h-hItem+vy}
	= [ipShift {vx=lMargin,vy=tMargin} item:done]
where
	pos	= item.ipPos


//	ItPos operations.

::	ItPos
	=	{	ipId		:: Id					// The ControlId
		,	ipPos		:: ItemPos				// The given ItemPos
		,	ipCorner	:: !Point				// Left top corner of item
		,	ipSize		:: Size					// Size of item
		,	ipDepends	:: [RCPos]				// Dependent items (this item is Renter/Corner item)
		,	ipOffset	:: ItemOffset			// The vector by which compound elements should be moved
		}

dummyItPos
	:==	{	ipId		= sysId 0
		,	ipPos		= (Left,zero)
		,	ipCorner	= zero
		,	ipSize		= zero
		,	ipDepends	= []
		,	ipOffset	= zero
		}

newItPos :: !Id !ItemPos !Size -> ItPos
newItPos id pos size
	= {	ipId		= id
	  ,	ipPos		= pos
	  ,	ipCorner	= zero
	  ,	ipSize		= size
	  ,	ipDepends	= []
	  ,	ipOffset	= zero
	  }

ipShift :: !ItemOffset !ItPos -> ItPos
ipShift offset itPos=:{ipCorner,ipDepends,ipOffset}
	= {	itPos &	ipCorner	= addPointVector offset ipCorner
			  ,	ipDepends	= map (cpShift offset) ipDepends
			  ,	ipOffset	= offset+ipOffset
	  }

ipGetItemRect :: !Id !ItPos -> (Point,Size)
ipGetItemRect id itPos=:{ipId}
	| id==ipId
	= (itPos.ipCorner,itPos.ipSize)
	= (cp.cpCorner,   cp.cpSize)
where
	(_,cp) = Select (cpEqId id) dummyCPos itPos.ipDepends

ipContainsId :: !Id ![ItPos] -> Bool
ipContainsId id [itPos:itPoss]
	= id==itPos.ipId || cpContainsId id itPos.ipDepends || ipContainsId id itPoss
ipContainsId _ _
	= False

ipRemoveItPos :: !Id ![ItPos] -> (!Bool,!ItPos,![ItPos])
ipRemoveItPos id [itPos:itPoss]
	| id==itPos.ipId
	= (True,	itPos, itPoss)
	| cpContainsId id itPos.ipDepends
	= (True,	itPos, itPoss)
	= (exists,	itPos1,[itPos:itPoss1])
	with
		(exists,itPos1,itPoss1)	= ipRemoveItPos id itPoss
ipRemoveItPos _ itPoss
	= (False,dummyItPos,itPoss)

ipRemoveIdRect :: !Id ![ItPos] -> (!Bool,!Point,!Size,!ItemOffset,![ItPos])
ipRemoveIdRect id items=:[ip:ips]
	| id==ipId && isEmpty depends
	= (isFixedPos,corner,size,  offset, ips)
	| id==ipId
	= (isFixedPos,corner,size,  offset, items)
	| inDepends
	= (isFixedPos,cpCorner,cpSize, offset1,[{ip & ipDepends=depends1}:ips])
	= (isFixedPos`,corner`,size`,offset`,[ip:ips`])
	with
		(isFixedPos`,corner`,size`,offset`,ips`) = ipRemoveIdRect id ips
where
	(inDepends,cpCorner,cpSize,offset1,depends1)
					= cpRemoveIdRect id depends
	ipId			= ip.ipId
	corner			= ip.ipCorner
	size			= ip.ipSize
	depends			= ip.ipDepends
	offset			= ip.ipOffset
	(isFixedPos,_)	= IsFix ip.ipPos
ipRemoveIdRect id _
	= layoutError "ipRemoveIdRect" ("Unknown ItemId "+++toString id)


//	RCPos operations.

::	RCPos
	=	{	cpId		:: Id					// The ControlId
		,	cpCorner	:: !Point				// Left top position of the item
		,	cpSize		:: Size					// Size of the item
		,	cpOffset	:: ItemOffset			// The vector by which compound elements should be moved
		}

dummyCPos
	:==	{	cpId		= sysId 0
		,	cpCorner	= zero
		,	cpSize		= zero
		,	cpOffset	= zero
		}

cpEqId :: !Id !RCPos -> Bool
cpEqId id {cpId} = id==cpId

cpShift :: !ItemOffset !RCPos -> RCPos
cpShift offset cp=:{cpCorner,cpOffset}
	= {	cp & cpCorner	= addPointVector offset cpCorner
		   , cpOffset	= offset+cpOffset
	  }

cpMaxXY :: !Size ![RCPos] -> Size
cpMaxXY {w=maxX,h=maxY} [cp:cps]
	= cpMaxXY {w=max maxX (corner.x+size.w),h=max maxY (corner.y+size.h)} cps
where
	corner	= cp.cpCorner
	size	= cp.cpSize
cpMaxXY max _
	= max

cpContainsId :: !Id ![RCPos] -> Bool
cpContainsId id [cp:cps]
	= id==cp.cpId || cpContainsId id cps
cpContainsId _ _
	= False

cpRemoveIdRect :: !Id ![RCPos] -> (!Bool,!Point,!Size,!ItemOffset,![RCPos])
cpRemoveIdRect id [cp:cps]
	| id==cp.cpId
	= (True,cp.cpCorner,cp.cpSize,cp.cpOffset,cps)
	= (found,corner,size,offset,[cp:cps1])
	with
		(found,corner,size,offset,cps1) = cpRemoveIdRect id cps
cpRemoveIdRect _ cps
	= (False,zero,zero,zero,cps)


//	ItemPos predicates.

IsFix :: !ItemPos -> (!Bool,!Point)
IsFix (Fix point,_)			= (True,point)
IsFix _						= (False,zero)

IsLine :: !ItemPos -> Bool
IsLine (Left,_)				= True
IsLine (Center,_)			= True
IsLine (Right,_)			= True
IsLine _					= False

IsRelative :: !ItemPos -> (!Bool,!Id)
IsRelative (LeftOf	id,_)	= (True,id)
IsRelative (RightTo	id,_)	= (True,id)
IsRelative (Above	id,_)	= (True,id)
IsRelative (Below	id,_)	= (True,id)
IsRelative _				= (False,sysId 0)

IsRelativeX :: !ItemPos		-> Bool
IsRelativeX (LeftOf	 _,_)	= True
IsRelativeX (RightTo _,_)	= True
IsRelativeX _				= False

IsRenter :: !ItemPos		-> Bool
IsRenter (Center,_)			= True
IsRenter (Right, _)			= True
IsRenter _					= False

IsCorner :: !ItemPos		-> Bool
IsCorner (LeftTop,    _)	= True
IsCorner (RightTop,	  _)	= True
IsCorner (LeftBottom, _)	= True
IsCorner (RightBottom,_)	= True
IsCorner _					= False

IsLeftOf :: !ItemPos		-> Bool
IsLeftOf (LeftOf _,_)		= True
IsLeftOf _					= False

IsBelow  :: !ItemPos		-> Bool
IsBelow  (Below _,_)		= True
IsBelow  _					= False
